package easik.sketch.util.Tex;


import java.awt.Point;
import java.awt.geom.Rectangle2D;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.util.LinkedList;
import java.util.Vector;


import org.jgraph.JGraph;
import org.jgraph.graph.DefaultEdge;
import org.jgraph.graph.DefaultGraphCell;
import org.jgraph.graph.DefaultGraphModel;
import org.jgraph.graph.GraphModel;
import org.jgraph.graph.AttributeMap.SerializableRectangle2D;

import easik.Easik;
import easik.sketch.constraint.Constraint;
import easik.sketch.edge.SketchEdge;
import easik.sketch.vertex.SketchVertex;

/**
 * Class to convert a JGraph object to a tex file.  This class generates an output file containing a description of the 
 * graph using formatting for GasTex.  The file contains no header information, and thus cannot be converted into a TeX file
 * as is.  
 * <br>
 * To convert the file into a readable TeX file, the user must add relevant header information, including a call to use
 * the package GasTeX.  This package is freely available on the web.
 * 
 * @author William Belanger 2006
 * @author Vera Ranieri 2006
 * @since 2006-07-10 William Belanger
 * @version 2006-07-12 Vera Ranieri
 *
 */
public class TexFile extends File {
	/**
	 * The TeX file extension
	 */
	private static final String EXTENSION = ".tex";
	/**
	 * The description of the file
	 */
	private static final String DESCRIPTION = "TEX image file";
	/**
	 * The size of the box for an entity
	 */
	private static final int BOXLENGTH = 60;
	/**
	 * The size of the constraint box
	 */
	private static final int CONSTRAINT_LENGTH = 30;
	
	
	/**
	 * A point to transform
	 */
	private Point _transform;
	/**
	 * A point to flip
	 */
	private Point _flip;
	
	/**
	 * Returns the extension of this file
	 * @return The extension .tex
	 */
	public static final String getExtension(){
		return EXTENSION;
	}
	
	/**
	 * Gets the description of this file
	 * @return The description 'TEX image file'.
	 */
	public static final String getDescription(){
		return DESCRIPTION;
	}
	
	/**
	 * Determines whether the file has the proper extension appeneded to the file
	 * @param fileName The name of the file
	 * @return True if it ends in the extension .tex, false otherwise.
	 */
	public static final boolean isOfType(String fileName){
		return fileName.endsWith(getExtension());
	}
	
	/**
	 * Determines whether the file is of the proper type.
	 * @param file The file to be investigated.
	 * @return true if the file is has .tex as its extension, false otherwise
	 */
	public static final boolean isOfType(File file){
		return isOfType(file.getName());
	}
	
	/**
	 * Determines whether the file has the correct extension
	 * @param extension The current extension of the file
	 * @return true if the extension is the proper extension, false otherwise
	 */
	public static final boolean isCorrectExtension(String extension){
		return getExtension().equals(extension) ||
			getExtension().equals("." + extension);
	}
	
	/**
	 * Constructor that calls the super constructor. It also sets each constraint to have a unique name
	 * @param fileLoc The absolute path of the file.
	 */
	public TexFile(String fileLoc){
		super(fileLoc);
		LinkedList<Constraint> constraints = Easik.getInstance().getFrame().getSketch().getConstraints();
		int count = 0;
		for(Constraint c : constraints){
			c.setName(c.toString()+count);
			count++;
		}
	}
	
	/**
	 * Construtor that calls TexFile(String fileLoc) constructor by determining the absolute path of the file passed.
	 * @param aFile The file to be made into a TexFile.
	 */
	public TexFile(File aFile){
		this(aFile.getAbsolutePath());
	}
	
	/**
	 * Determines the minimum point on the graph, to set the bounds of the GasTeX image.
	 * 
	 * @param graph The graph being converted into a GasTeX image
	 * @return A Point of the minimum point on the <it>graph</it> object.
	 */
	private static Point getMin(JGraph graph){
		Object[] obs = graph.getGraphLayoutCache().getCells(false, true, false, false); // Only Vertices
		Rectangle2D r = graph.getCellBounds(obs);
		return new Point((int)r.getMinX(), (int)r.getMinY());
	}
	
	/**
	 * Determines the maximum point on the graph, to set the bounds of the GasTeX image.
	 * 
	 * @param graph The graph being converted into a GasTeX image.
	 * @return A point of the maximum point on the <it>graph</it> object
	 */
	private static Point getMax(JGraph graph){
		Object[] obs = graph.getGraphLayoutCache().getCells(false, true, false, false); // Only Vertices
		Rectangle2D r = graph.getCellBounds(obs);
		return new Point((int)r.getMaxX(), (int)r.getMaxY());
	}

	/**
	 * Saves the graph as a \.tex file by converting all elements to strings formatted for GasTex.  Overloads the
	 * <it>save</it> function in the <b>File</b> class.
	 * 
	 * @param toSave The graph to convert to a GasTex image
	 * @throws IOException Write error
	 */
	public void save(JGraph toSave) throws IOException {
		_transform = getMin(toSave);
		_flip = getMax(toSave);
		GraphModel m = toSave.getModel();
		Vector<DefaultEdge> edges = new Vector<DefaultEdge>();
		String sep = System.getProperty("line.separator");
		Object[] cells = toSave.getGraphLayoutCache().getCells(true, true, false, true); // Vertices and Edges
		BufferedWriter bw = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(this)));
		beginWrite(bw);
		for(Object o : cells){
			if(DefaultGraphModel.isVertex(m, o)){ // o is a vertex
				writeVertex(bw, (DefaultGraphCell)o, toSave);
				bw.write(sep);
			}
			else{ // o is an edge
				edges.add((DefaultEdge)o); // Save this for later. Have to draw edges after all nodes are drawn.
			}
		}
		for(DefaultEdge e : edges){
			writeEdge(bw, e, toSave);
			bw.write(sep);
		}
		try{
			finishWrite(bw);
		}
		catch (IOException e)
		{
			System.out.println(e.toString());
			e.printStackTrace();
		}
		finally
		{
			try{
				bw.close();
			}
			catch (IOException e)
			{
				System.out.println(e.toString());
				e.printStackTrace();
			}
		}
	}

	/**
	 * Sets the initial conditions of the image.
	 * 
	 * @param bw The buffered writer being used to write to the output file
	 * @throws IOException Write error
	 */
	private void beginWrite(BufferedWriter bw) throws IOException {
		String sep = System.getProperty("line.separator");
		bw.write("\\begingroup"+ sep);
		bw.write("\\unitlength=1pt" + sep);
		bw.write("\\begin{picture}(" + _flip.x + ", " + _flip.y + ")" + sep);
		bw.write("\\gasset{Nw=" + BOXLENGTH + ",Nmr=0}" + sep);
	}
	
	/**
	 * Sets the final conditions of the image.
	 * 
	 * @param bw The buffered writer being used to write to the output file
	 * @throws IOException Write error
	 */
	private void finishWrite(BufferedWriter bw) throws IOException{
		bw.write("\\end{picture}");
		bw.write(System.getProperty("line.separator"));
		bw.write("\\endgroup");
		bw.close();
	}
	
	/**
	 * Writes the information about a vertex to the output file.  Vertices are defined as either an EntityNode, or a 
	 * Constraint.  Depending on which class the vertex belongs to, the write is done to distinguish between the 
	 * two objects.
	 * <br>
	 * EntityNodes are written as standard rectangles which have a minimum length of <it>BOXLENGTH</it>.  Constraints are
	 * drawn as standard rectangles shaded in gray, with the outline being dashed.
	 * 
	 * @param bw The BufferedWriter being used to write to this file
	 * @param vertex The vertex to addd to the output file
	 * @param graph The graph which is being converted to a GasTeX image
	 * @throws IOException Write error
	 */
	private void writeVertex(BufferedWriter bw, DefaultGraphCell vertex, JGraph graph) throws IOException{
		String sep = System.getProperty("line.separator");
		String node = "\\node";
		
		SketchVertex obj = (SketchVertex)graph.getModel().getValue(vertex);
		String name = obj.getName();
		
		if(name.length() > 20){
			bw.write("\\gasset{Nmr=0, Nw = "+ (BOXLENGTH + name.length()) +"}" + sep);
		}
		
		if(obj instanceof Constraint){
			bw.write("\\gasset{Nmr=0, Nw=" + CONSTRAINT_LENGTH +"}"+ sep);
			node += "[fillgray=0.85, dash={1}0]";
		}
		
		bw.write(node);
		bw.write("(" + name + ")");
		SerializableRectangle2D bounds = (SerializableRectangle2D)graph.getCellBounds(vertex);
		int px = ( (int)bounds.getCenterX() - _transform.x );
		int py = _flip.y - ( (int)bounds.getCenterY() - _transform.y );
		if(bounds != null){
			bw.write("(" + px + "," + py + ")");
		}
		if(obj instanceof Constraint){
			name = obj.toString();
		}
		bw.write("{$" + name +"$}");
		if(name.length() > 20 || obj instanceof Constraint){
			bw.write(sep + "\\gasset{Nw=" + BOXLENGTH+ ",Nmr=0}");
		}
	}
	
	/**
	 * Writes the information about an edge to the output file.  Edges are defined as either a SketchEdge, or a 
	 * GuideEdge.  Depending on which class the edge belongs to, the write is done to distinguish between the 
	 * two objects.
	 * <br>
	 * SketchEdges are written as standard lines leading from the source to the target.  If the SketchEdge is 
	 * injective, the edge is drawn appropriately.  GuideEdges are drawn as dashed lines leading from the source (the 
	 * Constraint) to the target (the EntitiyNode involved in the constraint). 
	 *  
	 * @param bw The BufferedWriter being used to write to this file.
	 * @param edge The edge which is being added to the GasTeX file
	 * @param graph The graph which is being converted to a GasTeX image.
	 * @throws IOException Write Error
	 */
	private void writeEdge(BufferedWriter bw, DefaultEdge edge, JGraph graph) throws IOException{
		DefaultGraphModel m = (DefaultGraphModel)graph.getModel();
		Object obj = m.getValue(edge);
		String edgeLabel, drawLoop, drawEdge;
		if(obj instanceof SketchEdge){
			SketchEdge e = (SketchEdge)m.getValue(edge);
			edgeLabel = e.getName();
			drawLoop = "\\drawloop(";
			drawEdge = "\\drawedge("; 
			if(e.getInjective()){
				drawLoop = "\\drawloop[ATnb=1,ATangle=140,ATLength=5,ATlength=5](";
				drawEdge = "\\drawedge[ATnb=1,ATangle=140,ATLength=5,ATlength=5](";
			}
				
		}
		else{
			edgeLabel = "";
			drawLoop = "\\drawloop[dash={5}4](";
			drawEdge = "\\drawedge[dash={5}4](";
		}
		
		String sourceLabel = ((SketchVertex)m.getValue(DefaultGraphModel.getSourceVertex(m, edge))).getName();
		String targetLabel = ((SketchVertex)m.getValue(DefaultGraphModel.getTargetVertex(m, edge))).getName();
		if(sourceLabel.equals(targetLabel)){
			bw.write( drawLoop + sourceLabel + "){$" + edgeLabel + "$}");
		}
		else{
			bw.write( drawEdge + sourceLabel + "," + targetLabel +"){$" + edgeLabel +"$}");
		}
		
	}
}
